# Staff Password Management & Profile Editing Guide

## Overview
Complete staff password management system with secure hashing, profile editing capabilities, and responsive design across all devices.

---

## Features Added

### 1. **Admin Staff Registration with Password**

#### Location
**Admin Dashboard** → **Staff Management** → **Add Staff Member**

#### Password Field
- **Required for new staff**: Min 8 characters
- **Optional when editing**: Leave blank to keep current password
- **Secure**: Uses bcrypt hashing (PASSWORD_BCRYPT)
- **Eye icon**: Toggle visibility to verify password entry

#### Form Fields
1. **Name** - Staff member full name
2. **Email** - Unique email address (required)
3. **Password** - Minimum 8 characters (required for new staff)
4. **Phone** - Contact number
5. **Position/Role** - Job title (e.g., Cleaner, Supervisor)
6. **Assigned Services** - Select multiple services
7. **Status** - Active/Inactive

#### Validation Rules
```
✓ Name: Required
✓ Email: Required, unique, must not already exist
✓ Password (Add): Required, minimum 8 characters
✓ Password (Edit): Optional, minimum 8 characters if provided
✓ Phone: Required
✓ Position: Required
```

---

## Staff Profile Management

### Access
Staff members can access profile editing in their **Staff Dashboard** → **My Profile** tab

### Profile Display
Shows current information:
- Staff avatar with initials
- Full name and position
- Email address
- Phone number
- Assigned services
- Account status (Active/Inactive)

### Profile Edit Modal
Accessible via **Edit Profile** button in My Profile tab

#### Editable Fields
1. **Email Address**
   - Must be unique
   - Cannot match another staff member's email
   - Validation error if duplicate

2. **Phone Number**
   - Update contact information

3. **Password Change**
   - Leave blank to keep current password
   - Must be minimum 8 characters if changing
   - Requires confirmation match
   - Eye icons to toggle visibility

#### Password Validation
```
✓ Minimum 8 characters
✓ Must match confirmation password
✓ Optional (leave blank to keep current)
✓ Secure bcrypt hashing applied
```

---

## Technical Implementation

### Database Structure

#### Staff Table
```sql
CREATE TABLE staff (
  id INT PRIMARY KEY AUTO_INCREMENT,
  name VARCHAR(255) NOT NULL,
  email VARCHAR(255) UNIQUE NOT NULL,
  phone VARCHAR(20) NOT NULL,
  position VARCHAR(100) NOT NULL,
  password VARCHAR(255) NOT NULL,  -- bcrypt hash
  status ENUM('active', 'inactive') DEFAULT 'active',
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
)
```

### API Endpoints

#### Add Staff Member
```
POST api_handler.php
{
  action: 'add_staff',
  name: 'John Doe',
  email: 'john@example.com',
  password: 'SecurePass123',  -- Required, min 8 chars
  phone: '555-1234',
  position: 'Cleaner',
  status: 'active',
  services: JSON.stringify([1, 2, 3])
}
```

#### Edit Staff Member
```
POST api_handler.php
{
  action: 'edit_staff',
  staff_id: 1,
  name: 'John Doe',
  email: 'john@example.com',
  password: 'NewPassword123',  -- Optional, min 8 chars if provided
  phone: '555-1234',
  position: 'Cleaner',
  status: 'active',
  services: JSON.stringify([1, 2])
}
```

#### Update Staff Profile
```
POST api_handler.php
{
  action: 'update_staff_profile',
  field: 'email' | 'phone' | 'password',
  value: 'new_value'
}
```

**Example Requests:**
```javascript
// Update email
{
  action: 'update_staff_profile',
  field: 'email',
  value: 'newemail@example.com'
}

// Update phone
{
  action: 'update_staff_profile',
  field: 'phone',
  value: '555-9876'
}

// Update password
{
  action: 'update_staff_profile',
  field: 'password',
  value: 'NewPassword123'
}
```

### API Responses

#### Success
```json
{
  "success": true,
  "message": "Staff member added successfully"
}
```

#### Error - Password Too Short
```json
{
  "success": false,
  "message": "Password must be at least 8 characters long"
}
```

#### Error - Email Exists
```json
{
  "success": false,
  "message": "Email already exists"
}
```

#### Error - Passwords Don't Match
```json
{
  "success": false,
  "message": "Passwords do not match"
}
```

---

## Security Features

### Password Protection
- ✅ **bcrypt hashing** - Industry-standard algorithm
- ✅ **Minimum 8 characters** - Prevents weak passwords
- ✅ **Confirmation matching** - UI validates before submission
- ✅ **Server-side validation** - Validates all inputs on backend
- ✅ **No plain text storage** - Passwords only stored as hashes

### Email Validation
- ✅ **Unique constraint** - Prevents duplicate emails
- ✅ **Database-level validation** - Enforced in schema
- ✅ **Application-level validation** - Checked before insertion
- ✅ **Duplicate detection on edit** - Won't allow email already used by others

### Data Protection
- ✅ **Session-based authentication** - Secure staff verification
- ✅ **HTTPS recommended** - For production environments
- ✅ **Input sanitization** - Prevents SQL injection
- ✅ **Type validation** - Ensures correct data types

---

## Responsive Design Implementation

### Desktop (769px and above)
- Full-width modals (max-width: 600px)
- Comfortable padding (30px)
- Side-by-side buttons
- Full field widths

### Tablet (600px - 768px)
- Modal width: 95% with max-width 600px
- Reduced padding (20px)
- Stacked buttons (flex-direction: column-reverse)
- Full-width buttons
- Optimized form spacing

### Mobile (Below 600px)
- Modal width: 90-95%
- Minimal padding (15px)
- Completely stacked layout
- Full-width buttons and inputs
- Larger touch targets (16px+ font)
- Modal header adjusted for small screens
- Form labels: 13px font size
- Reduced button padding (10px 15px)

### Responsive CSS Breakpoints
```css
/* Desktop */
@media (min-width: 769px) {
  .modal-content { padding: 30px; }
  .modal-footer { flex-direction: row; justify-content: flex-end; }
  .btn-modal-submit, .btn-modal-cancel { width: auto; }
}

/* Tablet */
@media (max-width: 768px) {
  .modal-content { padding: 20px; max-width: 95%; }
  .modal-footer { flex-direction: column-reverse; }
  .btn-modal-submit, .btn-modal-cancel { width: 100%; }
}

/* Mobile */
@media (max-width: 480px) {
  .modal-content { padding: 15px; margin: 10px; }
  .modal-header { flex-direction: column; align-items: flex-start; }
  .form-control { font-size: 16px; }
  .modal-close { align-self: flex-end; margin-top: -40px; }
}
```

---

## User Interface Components

### Password Field with Toggle
- Eye icon button to show/hide password
- Position: absolute right inside input
- Color: Primary green (#109c3d)
- Smooth transitions
- Touch-friendly on mobile

### Modal Layout
1. **Header**
   - Title on left
   - Close button (×) on right
   - Border-bottom separator

2. **Body**
   - Alert message display area
   - Form fields with labels
   - Password section with divider line
   - Help text and instructions

3. **Footer**
   - Cancel button (gray)
   - Submit button (green)
   - Responsive flex layout

### Alert Messages
- **Success**: Green background, primary color text, auto-hides after 3 seconds
- **Danger**: Red background, danger color text, persists until next action

---

## Setup Instructions

### 1. Database Setup
1. Run `setup_database.php`
2. Ensure staff table is created with password column
3. Verify staff_services and staff_bookings tables exist

### 2. Add First Staff Member
1. Log in to Admin Dashboard
2. Go to **Staff Management**
3. Click **Add Staff Member**
4. Fill in all required fields including password (min 8 characters)
5. Select assigned services
6. Click **Save Staff Member**

### 3. Staff Login
1. Staff navigate to `staff-login.php`
2. Enter email and password
3. Click **Login**

### 4. Staff Profile Management
1. Log in as staff member
2. Go to **My Profile** tab
3. Click **Edit Profile**
4. Update email, phone, or password
5. Click **Save Changes**

---

## Testing Checklist

### Password Functionality
- [ ] Add new staff member with password (min 8 chars)
- [ ] Edit staff member without changing password
- [ ] Edit staff member and change password
- [ ] Test password too short error
- [ ] Test password confirmation mismatch error
- [ ] Verify passwords don't appear in logs

### Profile Editing
- [ ] Update email address
- [ ] Test email already exists error
- [ ] Update phone number
- [ ] Change password in profile
- [ ] Test password confirmation matching
- [ ] Verify profile changes persist after logout/login

### Responsive Design
- [ ] Test on desktop (1920x1080)
- [ ] Test on laptop (1366x768)
- [ ] Test on tablet (768x1024)
- [ ] Test on mobile (375x667)
- [ ] Test on small mobile (320x568)
- [ ] Verify modal fits screen on all sizes
- [ ] Test button clicks on touch devices
- [ ] Verify no horizontal scrolling on mobile

### Security
- [ ] Passwords hashed in database (bcrypt)
- [ ] Can't log in with wrong password
- [ ] Can't access staff dashboard without login
- [ ] Can't register duplicate email
- [ ] Email validation on profile update
- [ ] Password validation on update

---

## Troubleshooting

### "Password must be at least 8 characters long"
**Issue**: User entered password less than 8 characters
**Solution**: Ensure password is at least 8 characters

### "Passwords do not match"
**Issue**: Password and confirm password fields don't match
**Solution**: Re-enter both passwords ensuring they match exactly

### "Email already exists"
**Issue**: Email is already used by another staff member
**Solution**: Use a different unique email address

### "Email already in use by another staff member"
**Issue**: Updated email conflicts with existing staff
**Solution**: Choose a different email address

### Modal not appearing
**Issue**: JavaScript error or modal not triggering
**Solution**:
- Clear browser cache
- Check browser console for errors
- Verify JavaScript is enabled
- Test in different browser

### Password field not showing/hiding
**Issue**: Eye icon toggle not working
**Solution**:
- Verify FontAwesome icons loaded
- Check JavaScript for togglePassword function
- Try in different browser

---

## Performance Notes

- Password hashing adds ~50-100ms to submission (normal bcrypt behavior)
- Email uniqueness check is very fast (indexed field)
- Profile updates are instantaneous
- No caching needed for real-time updates

---

## Future Enhancements

1. **Password Strength Meter** - Visual feedback during password entry
2. **Password History** - Prevent reuse of old passwords
3. **Two-Factor Authentication** - Additional security layer
4. **Password Reset Email** - Self-service password recovery
5. **Login History** - Track staff member logins
6. **Session Management** - Admin can force logout staff
7. **Audit Logging** - Track profile modifications
8. **Password Expiration** - Force password changes periodically

---

## Browser Compatibility

✅ Chrome 90+
✅ Firefox 88+
✅ Safari 14+
✅ Edge 90+
✅ Mobile browsers (iOS Safari, Chrome Mobile)

---

## Support & Maintenance

### Common Maintenance Tasks
1. **Reset Staff Password**: Admin can edit staff and set new password
2. **Update Staff Email**: Staff or admin can update via profile or admin panel
3. **Change Staff Status**: Admin can deactivate/reactivate staff member
4. **View Staff Login**: Check session data for staff_id and user_type

### Security Audit
- [ ] Verify all passwords are bcrypt hashes
- [ ] Check no plain text passwords in database
- [ ] Audit staff login attempts
- [ ] Verify session management working
- [ ] Test password validation rules

---

## Files Modified

- `admin-dashboard.php` - Added password field to Staff Modal
- `admin-dashboard.js` - Password validation and form handling
- `api_handler.php` - addStaff, editStaff, updateStaffProfile functions
- `staff-dashboard.php` - Profile edit modal and functions
- `setup_database.php` - Password column in staff table (already exists)

---

## Code Examples

### JavaScript - Update Profile
```javascript
function updateProfileField(field, value) {
    const formData = new FormData();
    formData.append('action', 'update_staff_profile');
    formData.append('field', field);
    formData.append('value', value);

    fetch('api_handler.php', { method: 'POST', body: formData })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showAlert('Success: ' + data.message);
            } else {
                showAlert('Error: ' + data.message);
            }
        });
}
```

### PHP - Password Hashing
```php
// When adding staff
$password = $_POST['password'];
$hashedPassword = password_hash($password, PASSWORD_BCRYPT);
// Store $hashedPassword in database

// When verifying login
if (password_verify($entered_password, $stored_hash)) {
    // Password is correct
}
```

---

## Changelog

### Version 1.0 (Current)
- ✅ Added password field to staff registration
- ✅ Implemented password hashing with bcrypt
- ✅ Added password validation (min 8 characters)
- ✅ Created staff profile edit modal
- ✅ Added email and phone update functionality
- ✅ Implemented password change capability
- ✅ Full responsive design for all screen sizes
- ✅ Password confirmation matching
- ✅ Email uniqueness validation
- ✅ Secure session management

